const { existsSync } = require('fs');
const { join, normalize } = require('path');

const $ = (function () {
	// eslint-disable-next-line no-undef
	if (typeof global !== 'undefined') return global;
	if (typeof self !== 'undefined') return self;
	if (typeof window !== 'undefined') return window;
	if (typeof globalThis !== 'undefined') return globalThis;
	throw new Error('unable to locate global object');
})();

if (!$?.defineCache) {
	$.defineCache = {};
}

const root = normalize(join(__dirname, '..', '..'));
const propertyCache = Symbol('propertyCache');

const namespace = {
	'@': root,
	'@platforms': join(root, 'platforms'),
	'@pkg': join(root, 'packages'),
	'@npm': join(root, 'packages', '@npm'),
	'@use': join(root, 'packages', '@usages'),
	'@ui': join(root, 'packages', '@ui'),
};

const properties = {
	'{arch}': process?.arch || 'x64',
	'{platform}': process?.platform || 'win32',
	'{type}': process?.type === 'browser' ? 'main' : 'render',
};

const bind = (obj, $context) => {
	if (!$context) {
		return obj;
	}

	if (obj?.prototype) {
		obj.prototype.$context = $context;
	}

	try {
		const _obj = typeof obj === 'function' ? obj.bind({ $context }) : obj;
		obj.$context = _obj.$context = $context;
		return _obj;
	} catch (e) {
		console.error(e);
	}

	return obj;
};

const packageJson = (path) => {
	const composed = join(root, 'packages', path);

	try {
		// TODO: existsSync may be deprecated
		return existsSync(join(composed, 'package.json')) ? composed : false;
	} catch (e) {}

	return false;
};

const dump = (obj) => {
	return JSON.stringify(obj, null, 4);
};

const compose = (name) => {
	if (Array.isArray(name)) {
		return name.map((_name) => compose(_name));
	}

	const parts = name.split('/').map((part) => {
		for (let prop in properties) {
			if (part.indexOf(prop) >= 0) part = part.replaceAll(prop, namespace[prop]);
		}

		return part;
	});

	if (parts?.[0] in namespace) {
		parts[0] = namespace[parts[0]];
	}

	return parts.join('/').replace(/\\/g, '/');
};

const resolve = (name) => {
	if (Array.isArray(name)) {
		return name.map((_name) => resolve(_name));
	}

	if (name in $.defineCache) {
		return $.defineCache?.[name]?.path;
	}

	const composed = compose(name);
	const pkg = packageJson(composed);

	try {
		const path = require.resolve(pkg || composed);
		$.defineCache[name] = { path };
	} catch (e) {
		console.error(e);
		throw new Error(`error resolve module: ${name}, at ${pkg || composed}, error ${e?.message || e}`);
	}

	return $.defineCache?.[name]?.path;
};

const property = (target, name, callable) => {
	Object.defineProperty(target, name, {
		configurable: false,
		enumerable: true,

		get() {
			if (!(propertyCache in target)) {
				target[propertyCache] = Object.create(null);
			}

			if (!(name in target[propertyCache])) {
				try {
					target[propertyCache][name] = callable();
				} catch (e) {
					console.error(e);
				}
			}

			return target[propertyCache]?.[name];
		},
	});
};

const sub = (path) => {
	return (name, context = null) => define(`${path}/${name}`, context);
};

const define = (name, context = null) => {
	if (Array.isArray(name)) {
		return name.map((_name) => define(_name, context));
	}

	if ($.defineCache?.[name]?.mod) {
		return $.defineCache[name].mod;
	}

	const path = resolve(name);

	if (!path) {
		throw new Error(`unable to resolve module: ${name}`);
	}

	$.defineCache[name].mod = bind(require(path), context);
	console.log('🧩 define module', `[${name}]`, path);

	return $.defineCache?.[name]?.mod;
};

define.property = property;
define.compose = compose;
define.bind = bind;
define.namespace = namespace;
define.properties = properties;
define.cache = $.defineCache;
define.resolve = resolve;
define.sub = sub;

module.exports = define;
